// Library Application
class Library {
    constructor() {
        this.books = [];
        this.filteredBooks = [];
        this.categories = new Set();
        this.ageRanges = new Set();
        this.currentBook = null;

        this.init();
    }

    async init() {
        await this.loadBooks();
        this.setupEventListeners();
        this.renderBooks();
        this.registerServiceWorker();
        this.hideLoading();
    }

    registerServiceWorker() {
        if ('serviceWorker' in navigator) {
            navigator.serviceWorker.register('./service-worker.js')
                .then(() => console.log('Library Service Worker registered'))
                .catch(err => console.log('Service Worker registration failed:', err));
        }
    }

    async loadBooks() {
        try {
            const response = await fetch('books/library-index.json');
            const data = await response.json();
            this.books = data.books || [];

            this.books.forEach(book => {
                if (book.category) this.categories.add(book.category);
                if (book.ageRange) this.ageRanges.add(book.ageRange);
            });

            this.filteredBooks = [...this.books];
            this.populateFilters();
            this.renderQuickFilters();
            this.updateStats();
        } catch (error) {
            console.error('Failed to load library:', error);
            this.showEmptyState();
        }
    }

    populateFilters() {
        const categoryFilter = document.getElementById('categoryFilter');
        const ageFilter = document.getElementById('ageFilter');

        // Preserve default option, remove previous dynamic ones
        categoryFilter.querySelectorAll('option[data-dynamic]').forEach(opt => opt.remove());
        ageFilter.querySelectorAll('option[data-dynamic]').forEach(opt => opt.remove());

        Array.from(this.categories).sort().forEach(category => {
            const option = document.createElement('option');
            option.value = category;
            option.textContent = category;
            option.dataset.dynamic = 'true';
            categoryFilter.appendChild(option);
        });

        Array.from(this.ageRanges).sort().forEach(age => {
            const option = document.createElement('option');
            option.value = age;
            option.textContent = age;
            option.dataset.dynamic = 'true';
            ageFilter.appendChild(option);
        });
    }

    setupEventListeners() {
        document.getElementById('searchInput').addEventListener('input', () => {
            this.filterBooks();
        });

        document.getElementById('categoryFilter').addEventListener('change', () => {
            const value = document.getElementById('categoryFilter').value;
            this.updateQuickFilterHighlight(value);
            this.filterBooks();
        });

        document.getElementById('ageFilter').addEventListener('change', () => {
            this.filterBooks();
        });

        document.getElementById('groupBy').addEventListener('change', () => {
            this.renderBooks();
        });

        document.getElementById('sortBy').addEventListener('change', () => {
            this.sortBooks();
            this.renderBooks();
        });

        document.getElementById('closeModalBtn').addEventListener('click', () => {
            this.closeModal();
        });

        document.querySelector('.modal-close').addEventListener('click', () => {
            this.closeModal();
        });

        document.querySelector('.modal-backdrop').addEventListener('click', () => {
            this.closeModal();
        });

        document.getElementById('openBookBtn').addEventListener('click', () => {
            if (this.currentBook) {
                window.location.href = `books/${this.currentBook.id}/index.html`;
            }
        });

        document.addEventListener('keydown', (event) => {
            if (event.key === 'Escape') {
                const modal = document.getElementById('bookModal');
                if (modal && modal.classList.contains('active')) {
                    this.closeModal();
                }
            }
        });
    }

    filterBooks() {
        const searchTerm = document.getElementById('searchInput').value.toLowerCase();
        const categoryFilter = document.getElementById('categoryFilter').value;
        const ageFilter = document.getElementById('ageFilter').value;

        this.filteredBooks = this.books.filter(book => {
            const matchesSearch = !searchTerm ||
                (book.title && book.title.toLowerCase().includes(searchTerm)) ||
                (book.author && book.author.toLowerCase().includes(searchTerm)) ||
                (book.series && book.series.toLowerCase().includes(searchTerm));

            const matchesCategory = !categoryFilter || book.category === categoryFilter;
            const matchesAge = !ageFilter || book.ageRange === ageFilter;

            return matchesSearch && matchesCategory && matchesAge;
        });

        this.sortBooks();
        this.updateQuickFilterHighlight(categoryFilter);
        this.renderBooks();
    }

    sortBooks() {
        const sortBy = document.getElementById('sortBy').value;

        switch (sortBy) {
            case 'title':
                this.filteredBooks.sort((a, b) => (a.title || '').localeCompare(b.title || ''));
                break;
            case 'author':
                this.filteredBooks.sort((a, b) => (a.author || '').localeCompare(b.author || ''));
                break;
            case 'recent':
                this.filteredBooks.sort((a, b) => new Date(b.dateAdded || 0) - new Date(a.dateAdded || 0));
                break;
            case 'reading-time':
                this.filteredBooks.sort((a, b) => {
                    const timeA = parseInt(a.readingTime, 10) || 0;
                    const timeB = parseInt(b.readingTime, 10) || 0;
                    return timeA - timeB;
                });
                break;
        }
    }

    renderBooks() {
        const bookGrid = document.getElementById('bookGrid');
        const emptyState = document.getElementById('emptyState');

        if (this.filteredBooks.length === 0) {
            bookGrid.style.display = 'none';
            emptyState.style.display = 'block';
            return;
        }

        bookGrid.style.display = 'grid';
        emptyState.style.display = 'none';
        bookGrid.innerHTML = '';

        const groupBy = document.getElementById('groupBy').value;

        if (groupBy === 'none') {
            this.filteredBooks.forEach(book => {
                bookGrid.appendChild(this.createBookCard(book));
            });
        } else {
            const groups = this.groupBooks(groupBy);

            Object.keys(groups).sort().forEach(groupName => {
                const heading = document.createElement('h3');
                heading.className = 'group-heading';
                heading.textContent = groupName;
                bookGrid.appendChild(heading);

                groups[groupName].forEach(book => {
                    bookGrid.appendChild(this.createBookCard(book));
                });
            });
        }
    }

    groupBooks(groupBy) {
        const groups = {};

        this.filteredBooks.forEach(book => {
            let groupName = 'Other';

            switch (groupBy) {
                case 'author':
                    groupName = book.author || 'Unknown Author';
                    break;
                case 'series':
                    groupName = book.series || 'Standalone Books';
                    break;
                case 'category':
                    groupName = book.category || 'Uncategorized';
                    break;
            }

            if (!groups[groupName]) {
                groups[groupName] = [];
            }
            groups[groupName].push(book);
        });

        return groups;
    }

    createBookCard(book) {
        const card = document.createElement('div');
        card.className = 'book-card';
        card.tabIndex = 0;
        card.setAttribute('role', 'button');
        card.setAttribute('aria-label', `Open ${book.title || 'book'}`);
        card.addEventListener('click', () => this.openBookModal(book));
        card.addEventListener('keypress', (event) => {
            if (event.key === 'Enter' || event.key === ' ') {
                event.preventDefault();
                this.openBookModal(book);
            }
        });

        const cover = document.createElement('img');
        cover.className = 'book-cover';
        cover.alt = book.title || 'Book cover';
        if (book.coverImage) {
            cover.src = `books/${book.id}/${book.coverImage}`;
        } else {
            cover.src = this.generateFallbackCover(book.title);
            cover.classList.add('book-cover--fallback');
        }
        cover.addEventListener('error', () => {
            cover.src = this.generateFallbackCover(book.title);
            cover.classList.add('book-cover--fallback');
        }, { once: true });

        const info = document.createElement('div');
        info.className = 'book-info';

        const titleEl = document.createElement('div');
        titleEl.className = 'book-title';
        titleEl.textContent = book.title || 'Untitled';

        const authorEl = document.createElement('div');
        authorEl.className = 'book-author';
        authorEl.textContent = book.author ? `by ${book.author}` : 'Unknown author';

        const meta = document.createElement('div');
        meta.className = 'book-meta';
        if (book.category) meta.appendChild(this.createBadge(book.category));
        if (book.ageRange) meta.appendChild(this.createBadge(book.ageRange));

        info.appendChild(titleEl);
        info.appendChild(authorEl);
        info.appendChild(meta);

        card.appendChild(cover);
        card.appendChild(info);

        return card;
    }

    openBookModal(book) {
        this.currentBook = book;
        const modal = document.getElementById('bookModal');
        const modalCover = document.getElementById('modalCover');
        const fallback = this.generateFallbackCover(book.title);

        if (book.coverImage) {
            modalCover.src = `books/${book.id}/${book.coverImage}`;
        } else {
            modalCover.src = fallback;
        }
        modalCover.alt = book.title || 'Book cover';
        modalCover.addEventListener('error', () => {
            modalCover.src = fallback;
        }, { once: true });

        document.getElementById('modalTitle').textContent = book.title || 'Untitled';
        document.getElementById('modalAuthor').textContent = book.author || 'Unknown author';
        document.getElementById('modalDescription').textContent = book.description || 'No description available.';

        const categoryBadge = document.getElementById('modalCategory');
        categoryBadge.textContent = book.category || 'Uncategorized';
        categoryBadge.style.display = book.category ? 'inline-block' : 'none';

        const ageBadge = document.getElementById('modalAge');
        ageBadge.textContent = book.ageRange || '';
        ageBadge.style.display = book.ageRange ? 'inline-block' : 'none';

        const timeBadge = document.getElementById('modalReadingTime');
        timeBadge.textContent = book.readingTime || '';
        timeBadge.style.display = book.readingTime ? 'inline-block' : 'none';

        modal.classList.add('active');
        modal.setAttribute('aria-hidden', 'false');
    }

    closeModal() {
        const modal = document.getElementById('bookModal');
        modal.classList.remove('active');
        modal.setAttribute('aria-hidden', 'true');
        this.currentBook = null;
    }

    updateStats() {
        document.getElementById('bookCount').textContent =
            `${this.books.length} Book${this.books.length !== 1 ? 's' : ''}`;
    }

    showEmptyState() {
        document.getElementById('bookGrid').style.display = 'none';
        document.getElementById('emptyState').style.display = 'block';
    }

    hideLoading() {
        setTimeout(() => {
            document.getElementById('loading').style.display = 'none';
        }, 400);
    }

    createBadge(label) {
        const badge = document.createElement('span');
        badge.className = 'badge';
        badge.textContent = label;
        return badge;
    }

    renderQuickFilters() {
        const container = document.getElementById('quickFilters');
        if (!container) return;

        container.innerHTML = '';
        const categories = Array.from(this.categories).sort((a, b) => a.localeCompare(b));

        container.appendChild(this.createFilterChip('All Books', ''));
        categories.forEach(category => {
            container.appendChild(this.createFilterChip(category, category));
        });

        this.updateQuickFilterHighlight(document.getElementById('categoryFilter').value);
    }

    createFilterChip(label, value) {
        const button = document.createElement('button');
        button.type = 'button';
        button.className = 'filter-chip';
        button.dataset.value = value;
        button.textContent = label;
        button.addEventListener('click', () => {
            document.getElementById('categoryFilter').value = value;
            this.updateQuickFilterHighlight(value);
            this.filterBooks();
        });
        return button;
    }

    updateQuickFilterHighlight(value) {
        const container = document.getElementById('quickFilters');
        if (!container) return;

        Array.from(container.querySelectorAll('.filter-chip')).forEach(chip => {
            chip.classList.toggle('is-active', chip.dataset.value === value);
        });
    }

    generateFallbackCover(title) {
        const safeTitle = encodeURIComponent((title || 'My Book').slice(0, 24));
        return `data:image/svg+xml;utf8,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 400 400'><linearGradient id='g' x1='0' x2='1' y1='0' y2='1'><stop offset='0%' stop-color='%232ecc71'/><stop offset='100%' stop-color='%233498db'/></linearGradient><rect width='400' height='400' fill='url(%23g)'/><text x='200' y='200' text-anchor='middle' fill='white' font-family='Arial' font-size='32' dy='0.35em'>${safeTitle}</text></svg>`;
    }
}

document.addEventListener('DOMContentLoaded', () => {
    new Library();
});
