// Page Turner - CSS 3D Transform based page flipping
class PageTurner {
    constructor(container, pages) {
        this.container = container;
        this.pages = Array.from(pages);
        this.currentPage = 0;
        this.animationEnabled = true; // Can be toggled via settings
        this.init();
    }

    init() {
        // Set up z-index for proper stacking (odd pages on top)
        this.pages.forEach((page, i) => {
            page.classList.add('page');
            page.setAttribute('data-page', i);

            if (i % 2 === 0) {
                // Even index = odd page number (right pages)
                page.style.zIndex = this.pages.length - i;
                page.classList.add('page-odd');
            } else {
                // Odd index = even page number (left pages)
                page.classList.add('page-even');
            }

            this.container.appendChild(page);
        });

        // Add click handlers
        this.pages.forEach((page, i) => {
            page.onclick = () => this.handlePageClick(i);
        });

        this.updatePageCounter();
        this.updateButtons();
        this.preloadVisibleVideos(); // Preload first visible pages
    }

    handlePageClick(pageIndex) {
        const page = this.pages[pageIndex];

        if (pageIndex % 2 === 0) {
            // Odd page (right side) - flip forward
            if (!page.classList.contains('flipped')) {
                this.flipPage(pageIndex, true);
            }
        } else {
            // Even page (left side) - flip backward
            if (page.classList.contains('flipped')) {
                this.flipPage(pageIndex - 1, false);
            }
        }
    }

    flipPage(pageIndex, forward) {
        if (pageIndex < 0 || pageIndex >= this.pages.length - 1) return;

        const rightPage = this.pages[pageIndex];
        const leftPage = this.pages[pageIndex + 1];

        if (forward) {
            rightPage.classList.add('flipped');
            leftPage.classList.add('flipped');
            this.currentPage = pageIndex + 2;
        } else {
            rightPage.classList.remove('flipped');
            leftPage.classList.remove('flipped');
            this.currentPage = pageIndex;
        }

        this.updatePageCounter();
        this.updateButtons();
        this.preloadVisibleVideos();
    }

    preloadVisibleVideos() {
        // Preload videos on currently visible pages (current spread + next spread)
        const visiblePages = [];

        // Current visible pages
        for (let i = this.currentPage; i < this.currentPage + 2 && i < this.pages.length; i++) {
            visiblePages.push(i);
        }

        // Next spread (preload ahead)
        for (let i = this.currentPage + 2; i < this.currentPage + 4 && i < this.pages.length; i++) {
            visiblePages.push(i);
        }

        visiblePages.forEach(pageIndex => {
            const page = this.pages[pageIndex];
            if (!page) return;

            const videos = page.querySelectorAll('video[preload="metadata"]');
            videos.forEach(video => {
                // Upgrade to auto preload for visible/upcoming pages
                video.preload = 'auto';
                if (video.readyState < 2) {
                    video.load();
                }
            });
        });
    }

    nextPage() {
        // Find the next unflipped odd page
        for (let i = this.currentPage; i < this.pages.length; i += 2) {
            if (!this.pages[i].classList.contains('flipped')) {
                this.flipPage(i, true);
                return;
            }
        }
    }

    prevPage() {
        // Find the previous flipped odd page
        for (let i = this.currentPage - 2; i >= 0; i -= 2) {
            if (this.pages[i].classList.contains('flipped')) {
                this.flipPage(i, false);
                return;
            }
        }
    }

    getCurrentPage() {
        return this.currentPage;
    }

    getTotalPages() {
        return this.pages.length;
    }

    setAnimationEnabled(enabled) {
        this.animationEnabled = enabled;
        this.container.classList.toggle('no-animation', !enabled);
    }

    updatePageCounter() {
        const counter = document.getElementById('pageCounter');
        if (counter) {
            const displayPage = Math.floor(this.currentPage / 2) + 1;
            const totalPages = Math.ceil(this.pages.length / 2);
            counter.textContent = `${displayPage} / ${totalPages}`;
        }
    }

    updateButtons() {
        const prevBtn = document.getElementById('prevBtn');
        const nextBtn = document.getElementById('nextBtn');
        const resetBtn = document.getElementById('resetBtn');

        if (prevBtn) {
            prevBtn.disabled = this.currentPage === 0;
        }
        if (nextBtn) {
            // Check if there are more pages to flip
            let hasMore = false;
            for (let i = this.currentPage; i < this.pages.length; i += 2) {
                if (!this.pages[i].classList.contains('flipped')) {
                    hasMore = true;
                    break;
                }
            }
            nextBtn.disabled = !hasMore;
        }

        if (resetBtn) {
            resetBtn.disabled = this.currentPage === 0;
        }
    }

    pauseAllVideos() {
        this.pages.forEach(page => {
            const videos = page.querySelectorAll('video');
            videos.forEach(video => {
                if (!video.paused) {
                    video.pause();
                }
                if (video.currentTime !== 0) {
                    video.currentTime = 0;
                }
            });
        });
    }

    reset() {
        this.pages.forEach(page => {
            page.classList.remove('flipped');
        });
        this.currentPage = 0;
        this.pauseAllVideos();
        this.updatePageCounter();
        this.updateButtons();
        this.preloadVisibleVideos();
    }
}
