// Settings module - manages user preferences with localStorage persistence
class BookSettings {
    constructor() {
        this.settings = {
            muteVideos: false,
            hideNavigation: false,
            instantPages: false
        };

        this.loadSettings();
        this.initUI();
        this.initHotCorner();
    }

    loadSettings() {
        const saved = localStorage.getItem('bookSettings');
        if (saved) {
            try {
                this.settings = { ...this.settings, ...JSON.parse(saved) };
            } catch (e) {
                console.warn('Failed to load settings:', e);
            }
        }
    }

    saveSettings() {
        localStorage.setItem('bookSettings', JSON.stringify(this.settings));
    }

    initUI() {
        // Mute videos toggle
        const muteCheckbox = document.getElementById('muteVideos');
        if (muteCheckbox) {
            muteCheckbox.checked = this.settings.muteVideos;
            muteCheckbox.addEventListener('change', (e) => {
                this.settings.muteVideos = e.target.checked;
                this.saveSettings();
                this.applySettings();
            });
        }

        // Hide navigation toggle
        const hideNavCheckbox = document.getElementById('hideNavigation');
        if (hideNavCheckbox) {
            hideNavCheckbox.checked = this.settings.hideNavigation;
            hideNavCheckbox.addEventListener('change', (e) => {
                this.settings.hideNavigation = e.target.checked;
                this.saveSettings();
                this.applySettings();
            });
        }

        // Instant pages toggle
        const instantCheckbox = document.getElementById('instantPages');
        if (instantCheckbox) {
            instantCheckbox.checked = this.settings.instantPages;
            instantCheckbox.addEventListener('change', (e) => {
                this.settings.instantPages = e.target.checked;
                this.saveSettings();
                this.applySettings();
            });
        }

        // Apply initial settings
        this.applySettings();
    }

    initHotCorner() {
        const hotCorner = document.getElementById('settingsHotCorner');
        const settingsPanel = document.getElementById('settingsPanel');
        let touchTimer = null;
        let isHolding = false;

        // Touch and hold
        hotCorner.addEventListener('touchstart', (e) => {
            e.preventDefault();
            isHolding = true;
            hotCorner.classList.add('active');

            touchTimer = setTimeout(() => {
                if (isHolding) {
                    this.toggleSettings();
                }
            }, 1000); // 1 second hold
        });

        hotCorner.addEventListener('touchend', () => {
            isHolding = false;
            clearTimeout(touchTimer);
            hotCorner.classList.remove('active');
        });

        hotCorner.addEventListener('touchmove', () => {
            isHolding = false;
            clearTimeout(touchTimer);
            hotCorner.classList.remove('active');
        });

        // Mouse version for desktop testing
        hotCorner.addEventListener('mousedown', (e) => {
            isHolding = true;
            hotCorner.classList.add('active');

            touchTimer = setTimeout(() => {
                if (isHolding) {
                    this.toggleSettings();
                }
            }, 1000);
        });

        hotCorner.addEventListener('mouseup', () => {
            isHolding = false;
            clearTimeout(touchTimer);
            hotCorner.classList.remove('active');
        });

        hotCorner.addEventListener('mouseleave', () => {
            isHolding = false;
            clearTimeout(touchTimer);
            hotCorner.classList.remove('active');
        });

        // Swipe down gesture
        let touchStartY = 0;
        hotCorner.addEventListener('touchstart', (e) => {
            touchStartY = e.touches[0].clientY;
        }, { passive: true });

        hotCorner.addEventListener('touchmove', (e) => {
            const touchY = e.touches[0].clientY;
            const deltaY = touchY - touchStartY;

            if (deltaY > 50) { // Swipe down threshold
                this.toggleSettings();
                touchStartY = touchY; // Reset to prevent multiple triggers
            }
        }, { passive: true });

        // Close panel when clicking outside
        document.addEventListener('click', (e) => {
            if (settingsPanel.classList.contains('open') &&
                !settingsPanel.contains(e.target) &&
                !hotCorner.contains(e.target)) {
                this.closeSettings();
            }
        });
    }

    toggleSettings() {
        const panel = document.getElementById('settingsPanel');
        panel.classList.toggle('open');
    }

    closeSettings() {
        const panel = document.getElementById('settingsPanel');
        panel.classList.remove('open');
    }

    applySettings() {
        // Mute videos
        if (this.settings.muteVideos) {
            this.muteAllVideos();
        }

        // Hide navigation
        const controls = document.getElementById('controls');
        if (controls) {
            controls.style.display = this.settings.hideNavigation ? 'none' : 'flex';
        }

        // Instant pages (no animation)
        const pagesContainer = document.querySelector('.pages');
        if (pagesContainer) {
            pagesContainer.classList.toggle('no-animation', this.settings.instantPages);
        }

        // Also update page turner if it exists
        if (window.pageTurner && window.pageTurner.setAnimationEnabled) {
            window.pageTurner.setAnimationEnabled(!this.settings.instantPages);
        }
    }

    muteAllVideos() {
        const videos = document.querySelectorAll('video');
        videos.forEach(video => {
            video.muted = this.settings.muteVideos;
        });
    }

    getSettings() {
        return { ...this.settings };
    }
}

// Initialize settings when DOM is ready
let bookSettings;
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        bookSettings = new BookSettings();
    });
} else {
    bookSettings = new BookSettings();
}
