// Layout Engine - Renders pages from book data
class LayoutEngine {
    // Helper: Apply image controls (fit, position, zoom)
    static applyImageControls(img, imageControls) {
        if (!imageControls) {
            return;
        }

        const sourceName = img.src ? img.src.split('/').pop() : '[inline media]';
        console.log('Applying image controls:', imageControls, 'to', sourceName);

        // Apply styles BEFORE PageFlip loads (no need for !important)
        if (imageControls.fit) {
            img.style.objectFit = imageControls.fit;
        }
        if (imageControls.position) {
            img.style.objectPosition = imageControls.position;
        }
        if (imageControls.zoom && imageControls.zoom !== 100) {
            const scale = imageControls.zoom / 100;
            img.style.transform = `scale(${scale})`;
            // Ensure transform doesn't break layout
            img.style.transformOrigin = 'center center';
        }
    }

    // Helper: Create video element that shows first frame and plays on click
    // If imageSrc is provided, use it as poster; otherwise video first frame is used
    static createVideoElement(videoSrc, imageSrc, className, imageControls) {
        const video = document.createElement('video');
        video.src = 'assets/videos/' + videoSrc;
        video.className = className;
        video.preload = 'metadata'; // Load first frame only
        video.playsInline = true;
        video.loop = false;
        video.style.cursor = 'pointer';
        video.title = 'Tap to play';

        // Use image as poster if provided
        if (imageSrc) {
            video.poster = LayoutEngine.resolveImagePath(imageSrc);
        }

        // Check settings for mute preference
        video.muted = window.bookSettings ? window.bookSettings.getSettings().muteVideos : false;

        // Apply image controls to video
        LayoutEngine.applyImageControls(video, imageControls);

        // Click to play/pause
        video.addEventListener('click', (e) => {
            e.stopPropagation(); // Don't trigger page navigation

            if (video.paused) {
                // First play - preload if needed
                if (video.readyState < 3) {
                    video.preload = 'auto';
                    video.load();
                }

                video.play().catch(err => {
                    console.error('Video play failed:', err);
                });
            } else {
                video.pause();
            }
        });

        // When video ends, reset to first frame
        video.addEventListener('ended', () => {
            video.pause();
            video.currentTime = 0;
        });

        // Handle errors gracefully
        video.addEventListener('error', (e) => {
            console.error('Video load error:', videoSrc, e);
            video.title = 'Video unavailable';
            video.style.cursor = 'not-allowed';
        });

        return video;
    }

    // Helper: Create image element (when no video)
    static createImageElement(imageSrc, className, imageControls) {
        const img = document.createElement('img');
        img.src = LayoutEngine.resolveImagePath(imageSrc);
        img.className = className;
        LayoutEngine.applyImageControls(img, imageControls);
        return img;
    }

    // Helper: Create media element (video if available, otherwise image)
    static createMediaElement(pageData, className, pageNumber) {
        if (!pageData.components.image && !pageData.components.video) {
            return null;
        }

        let mediaElement;
        if (pageData.components.video) {
            // Use video with image as poster (or video first frame if no image)
            mediaElement = LayoutEngine.createVideoElement(
                pageData.components.video,
                pageData.components.image,
                className,
                pageData.imageControls
            );
        } else {
            // Image only
            mediaElement = LayoutEngine.createImageElement(
                pageData.components.image,
                className,
                pageData.imageControls
            );
            mediaElement.alt = `Page ${pageNumber}`;
        }
        return mediaElement;
    }

    static resolveImagePath(imageSrc) {
        if (!imageSrc) {
            return null;
        }
        if (imageSrc.startsWith('http') || imageSrc.startsWith('data:')) {
            return imageSrc;
        }
        if (imageSrc.startsWith('assets/')) {
            return imageSrc;
        }
        return 'assets/images/' + imageSrc.replace(/^\/+/, '');
    }

    static applySpreadControls(pageEl, imageControls, side) {
        delete pageEl.dataset.spreadFit;
        delete pageEl.dataset.spreadPosition;
        delete pageEl.dataset.spreadZoom;

        const defaults = {
            width: 200,
            height: 100,
            position: side === 'right' ? 'right center' : 'left center'
        };

        let sizeValue = `${defaults.width}% ${defaults.height}%`;
        let positionValue = defaults.position;

        if (imageControls) {
            if (imageControls.fit) {
                pageEl.dataset.spreadFit = imageControls.fit;
                switch (imageControls.fit) {
                    case 'fill':
                        sizeValue = '100% 100%';
                        break;
                    case 'contain':
                    case 'scale-down':
                        sizeValue = 'contain';
                        break;
                    case 'none':
                        sizeValue = 'auto';
                        break;
                    case 'cover':
                    case 'auto':
                    default:
                        sizeValue = `${defaults.width}% ${defaults.height}%`;
                        break;
                }
            }

            if (imageControls.position) {
                positionValue = imageControls.position;
                pageEl.dataset.spreadPosition = imageControls.position;
            }

            if (imageControls.zoom && imageControls.zoom !== 100) {
                const scale = imageControls.zoom / 100;
                const numericSizeMatch = sizeValue.match(/^(\d+(?:\.\d+)?)%\s+(\d+(?:\.\d+)?)%$/);
                if (numericSizeMatch) {
                    const width = parseFloat(numericSizeMatch[1]) * scale;
                    const height = parseFloat(numericSizeMatch[2]) * scale;
                    sizeValue = `${width}% ${height}%`;
                    pageEl.dataset.spreadZoom = String(imageControls.zoom);
                }
            }
        }

        pageEl.style.backgroundSize = sizeValue;
        pageEl.style.backgroundPosition = positionValue;
    }

    static setupSpreadPage(pageEl, pageData, side) {
        pageEl.classList.add('double-page-spread');
        if (side === 'right') {
            pageEl.classList.add('double-page-right');
            pageEl.classList.remove('double-page-left');
        } else {
            pageEl.classList.add('double-page-left');
            pageEl.classList.remove('double-page-right');
        }

        pageEl.dataset.spreadSide = side;

        const components = pageData && pageData.components ? pageData.components : {};
        const imageSrc = LayoutEngine.resolveImagePath(components.image);
        if (imageSrc) {
            const backgroundValue = 'url("' + imageSrc + '")';
            pageEl.dataset.spreadImage = imageSrc;
            pageEl.style.setProperty('--spread-image', backgroundValue);
            pageEl.style.backgroundImage = backgroundValue;
        } else {
            delete pageEl.dataset.spreadImage;
            pageEl.style.removeProperty('--spread-image');
            pageEl.style.removeProperty('background-image');
        }

        const defaultPosition = side === 'right' ? 'right center' : 'left center';
        pageEl.style.backgroundSize = '200% 100%';
        pageEl.style.backgroundPosition = defaultPosition;

        const imageControls = pageData && pageData.imageControls ? pageData.imageControls : null;
        LayoutEngine.applySpreadControls(pageEl, imageControls, side);
    }

    static renderPage(pageData, pageNumber) {
        const pageEl = document.createElement('div');
        pageEl.className = 'page';
        // Set page density for consistent flip animation (hard = stiff page turn)
        pageEl.setAttribute('data-density', 'hard');

        const content = document.createElement('div');
        content.className = 'page-content';

        switch (pageData.layout) {
            case 'full-image':
                const media = LayoutEngine.createMediaElement(pageData, 'page-image', pageNumber);
                if (media) content.appendChild(media);
                break;

            case 'double-page-spread':
                LayoutEngine.setupSpreadPage(pageEl, pageData, 'left');
                break;

            case 'image-top-text-bottom':
                content.className = 'page-content layout-vertical';
                const topMedia = LayoutEngine.createMediaElement(pageData, 'page-image-half', pageNumber);
                if (topMedia) content.appendChild(topMedia);
                if (pageData.components.text) {
                    const textDiv = document.createElement('div');
                    textDiv.className = 'page-text-half';
                    textDiv.innerHTML = pageData.components.text;
                    content.appendChild(textDiv);
                }
                break;

            case 'image-bottom-text-top':
                content.className = 'page-content layout-vertical';
                if (pageData.components.text) {
                    const textDiv = document.createElement('div');
                    textDiv.className = 'page-text-half';
                    textDiv.innerHTML = pageData.components.text;
                    content.appendChild(textDiv);
                }
                const bottomMedia = LayoutEngine.createMediaElement(pageData, 'page-image-half', pageNumber);
                if (bottomMedia) content.appendChild(bottomMedia);
                break;

            case 'image-left-text-right':
                content.className = 'page-content layout-horizontal';
                const leftMedia = LayoutEngine.createMediaElement(pageData, 'page-image-half', pageNumber);
                if (leftMedia) content.appendChild(leftMedia);
                if (pageData.components.text) {
                    const textDiv = document.createElement('div');
                    textDiv.className = 'page-text-half';
                    textDiv.innerHTML = pageData.components.text;
                    content.appendChild(textDiv);
                }
                break;

            case 'image-right-text-left':
                content.className = 'page-content layout-horizontal';
                if (pageData.components.text) {
                    const textDiv = document.createElement('div');
                    textDiv.className = 'page-text-half';
                    textDiv.innerHTML = pageData.components.text;
                    content.appendChild(textDiv);
                }
                const rightMedia = LayoutEngine.createMediaElement(pageData, 'page-image-half', pageNumber);
                if (rightMedia) content.appendChild(rightMedia);
                break;

            case 'text-only':
                if (pageData.components.text) {
                    const textDiv = document.createElement('div');
                    textDiv.className = 'page-text';
                    textDiv.innerHTML = pageData.components.text;
                    content.appendChild(textDiv);
                }
                break;

            case 'blank':
                // Empty page
                break;
        }

        pageEl.appendChild(content);

        // Apply page-level style overrides if they exist
        if (pageData.style) {
            if (pageData.style.backgroundColor) {
                pageEl.style.backgroundColor = pageData.style.backgroundColor;
            }
            if (pageData.style.textColor) {
                const textElements = pageEl.querySelectorAll('.page-text, .page-text-half');
                textElements.forEach(el => {
                    el.style.color = pageData.style.textColor;
                });
            }
            if (pageData.style.textAlign) {
                const textElements = pageEl.querySelectorAll('.page-text, .page-text-half');
                textElements.forEach(el => {
                    el.style.textAlign = pageData.style.textAlign;
                });
            }
        }

        return pageEl;
    }
}
